import template from './tradetracker-configuration.html.twig';

const { Component, Mixin, Defaults } = Shopware;
const { Criteria } = Shopware.Data;

Component.register('tradetracker-configuration', {
    template,

    mixins: [
        Mixin.getByName('notification'),
    ],

    data() {
        return {
            isLoading: false,
            isSaveSuccessful: false,
            domainId : null,
            apiConfig : null,
            config : null,
            salesChannels: [],
            productFeedData : null,
            productUrl : null,
            brandParam:[{id:'manufacture',name:'Manufacture'}],
            colorParam:[{id:'yes',name:'Yes'},{id:'no',name:'No'}],
            descriptionParam:[{id:'yes',name:'Yes'},{id:'no',name:'No'}],
            useParentImageParam:[{id:'yes',name:'Yes'},{id:'no',name:'No'}],
            propertyGroup : [],
            sortDirection: 'ASC',
            sortBy: 'name'
        }
    },

    created() {
        this.createdComponent();
        this.getProductFeed();
        this.getProductProperties();
    },

    inject: [
        'repositoryFactory',
        'configService',
        'acl',
    ],

    computed: {
        salesChannelRepository() {
            return this.repositoryFactory.create('sales_channel');
        },

        propertyRepository() {
            return this.repositoryFactory.create('property_group');
        },

        productFeedRepository() {
            return this.repositoryFactory.create('magmodules_product_feed');
        },
    },

    watch: {
        config: {
            handler: function() {
                this.domainId = this.$refs.configComponent.selectedSalesChannelId;
                this.productFeedData = null;
                this.getProductFeed();
            },
            deep: true
        }
    },

    methods: {
        createdComponent() {
            this.isLoading = true;
            const criteria = new Criteria();
            criteria.addFilter(
                Criteria.equalsAny('typeId', [
                    Defaults.storefrontSalesChannelTypeId,
                    Defaults.apiSalesChannelTypeId
                ])
            );

            this.salesChannelRepository.search(criteria, Shopware.Context.api).then(res => {
                res.add({id: null,
                    translated: {
                        name: this.$tc('sw-sales-channel-switch.labelDefaultOption')
                    }
                });
                this.salesChannels = res;
            }).finally(() => {
                this.isLoading = false;
            });
        },

        async getProductFeed() {
            const criteria = new Criteria();
            criteria.addFilter(Criteria.equals('salesChannelId', this.domainId));
            criteria.addSorting(Criteria.sort('createdAt', 'DESC'));
            criteria.setLimit(1);
            await this.productFeedRepository.search(criteria, Shopware.Context.api)
                .then(async (entity) => {
                    this.productFeedData = entity[0];
                     this.productUrl = await entity[0].productFeedUrl;
                });
        },

        onSave() {
            this.isLoading = true;
            this.$refs.configComponent.save().then(() => {
                this.isSaveSuccessful = true;
                this.createNotificationSuccess({
                    title: this.$tc('trade-tracker-configuration.action.titleSaveSuccess'),
                    message: this.$tc('trade-tracker-configuration.action.messageSaveSuccess')
                });
            }).finally(() => {
                this.isLoading = false;
            });
        },

        useParentImage(useParentImageParam) {
            return useParentImageParam;
        },

        color(colorParam) {
            return colorParam;
        },

        checkTextFieldInheritance(value) {
            if (typeof value !== 'string') {
                return true;
            }

            return value.length <= 0;
        },

        getProductProperties() {
            const criteria = new Criteria();
            criteria.addSorting(Criteria.sort(this.sortBy, this.sortDirection, this.useNaturalSorting));
            criteria.addAssociation('options');

            this.propertyRepository.search(criteria, Shopware.Context.api)
                .then((entity) => {
                    entity.forEach((name) => {
                        this.propertyGroup.push(name);
                    });
                    return this.propertyGroup;
                });
        },

        brand(brandParam) {
            return brandParam;
        },

        onSyncProduct(salesChannelMainId) {
            let data = new FormData();
            data.append('salesChannelId', salesChannelMainId);
            let headers = this.configService.getBasicHeaders();

            return this.configService.httpClient.post('/tradetracker/feedproduct', data, {headers}).then((response) => {
                if (response.data.type === 'Success') {
                    this.getProductFeed();
                    this.createNotificationSuccess({
                        title: this.$tc('trade-tracker-configuration.action.titleProductFeedSuccess'),
                        message: this.$tc('trade-tracker-configuration.action.messageProductFeedSuccess')
                    });
                } else {
                    this.createNotificationError({
                        title: this.$tc('trade-tracker-configuration.action.titleProductFeedFailed'),
                        message: this.$tc('trade-tracker-configuration.action.messageProductFeedFailed')
                    });
                }
            }).finally(() => {
                this.isLoading = false;
            });
        },
    }
});