<?php
declare(strict_types=1);

namespace Magmodules\TradeTracker\Storefront\Controller;

use Magmodules\TradeTracker\Service\Logger\TradeTrackerLogger;
use Psr\Log\LoggerInterface;
use Shopware\Core\Checkout\Customer\SalesChannel\AbstractRegisterRoute;
use Shopware\Core\Checkout\Customer\SalesChannel\AccountService;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepositoryInterface;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Routing\Exception\MissingRequestParameterException;
use Shopware\Core\Framework\Uuid\Uuid;
use Shopware\Core\Framework\Validation\DataBag\RequestDataBag;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Shopware\Storefront\Controller\StorefrontController;
use Shopware\Core\Framework\Routing\Annotation\RouteScope;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Routing\Annotation\Route;
use RuntimeException;

/**
 * @RouteScope(scopes={"storefront"})
 */
class RedirectController extends StorefrontController
{
    public const TRACKBACK_URL = 'https://tc.tradetracker.net/?c=%s&m=%s&a=%s&r=%s&u=%s';

    /**
     * @var Session
     */
    public $session;

    /**
     * @var SystemConfigService
     */
    public $systemConfigService;

    /**
     * @var AbstractRegisterRoute
     */
    public $registerRoute;

    /**
     * @var EntityRepositoryInterface
     */
    public $salutationRepository;

    /**
     * @var EntityRepositoryInterface
     */
    public $countryRepository;

    /**
     * @var EntityRepositoryInterface
     */
    public $customerRepository;

    /**
     * @var AccountService
     */
    public $accountService;

    /**
     * @var RouterInterface
     */
    public $router;

    /**
     * @var LoggerInterface
     */
    public $logger;

    /**
     * @var TradeTrackerLogger
     */
    public $tradeTrackerLogger;

    public function __construct(
        Session $session,
        SystemConfigService $systemConfigService,
        AbstractRegisterRoute $registerRoute,
        EntityRepositoryInterface $salutationRepository,
        EntityRepositoryInterface $countryRepository,
        EntityRepositoryInterface $customerRepository,
        AccountService $accountService,
        RouterInterface $router,
        LoggerInterface $logger,
        TradeTrackerLogger $tradeTrackerLogger
    ) {
        $this->session              = $session;
        $this->systemConfigService  = $systemConfigService;
        $this->registerRoute = $registerRoute;
        $this->salutationRepository = $salutationRepository;
        $this->countryRepository = $countryRepository;
        $this->customerRepository = $customerRepository;
        $this->accountService = $accountService;
        $this->router = $router;
        $this->logger = $logger;
        $this->tradeTrackerLogger = $tradeTrackerLogger;
    }

    /**
     * @Route("/tradetracker-redirect",
     *     name="storefront.trade-tracker.redirect",
     *     methods={"GET", "POST"}
     * )
     * @param SalesChannelContext $salesChannelContext
     * @param Request $request
     * @return RedirectResponse
     */
    public function tradeTrackerRedirect(
        SalesChannelContext $salesChannelContext,
        Request             $request
    ):RedirectResponse {
        $trackingGroupID = '';
        $directLinkingEnable = $this->systemConfigService->get(
            'MagmodulesTradeTracker.config.directLinkingEnable',
            $salesChannelContext->getSalesChannel()->getId()
        );
        $tt = $request->request->get('tt');
        if (!$tt || $directLinkingEnable === false) {
            return new RedirectResponse('/');
        }

        $redirectURL = $request->request->get('r');
        $domainName = $request->server->get('APP_URL');
        $domainName = empty(parse_url($domainName)['host']) ? null : '.' . parse_url($domainName)['host'];

        $trackingParam = explode('_', $tt);
        $campaignID = $trackingParam[0] ?? '';
        $materialID = $trackingParam[1] ?? '';
        $affiliateID = $trackingParam[2] ?? '';
        $reference = $trackingParam[3] ?? '';
        $time = strtotime(date('Y-m-d H:i:s'));
        $expireTime = time() + 31536000;

        // Calculate checksum.
        $encrypt = implode('::', ['CHK_', $campaignID, $materialID, $affiliateID, $reference]);
        $checkSum = base64_encode($encrypt);

        // Set tracking data.
        $trackingData = $materialID . '::' . $affiliateID . '::' . $reference . '::' . $checkSum . '::' . $time;

        // Set regular tracking cookie.
        setcookie('TT2_' . $campaignID, $trackingData, $expireTime, '/', $domainName);

        // Set session tracking cookie.
        setcookie('TTS_' . $campaignID, $trackingData, 0, '/', $domainName);

        // Set tracking group cookie.
        if (!empty($trackingGroupID)) {
            setcookie('__tgdat' . $trackingGroupID, $trackingData . '_' . $campaignID, $expireTime, '/', $domainName);
        }

        // Set track-back URL.
        $trackBackURL = sprintf(
            self::TRACKBACK_URL,
            $campaignID,
            $materialID,
            $affiliateID,
            urlencode($reference),
            urlencode($redirectURL)
        );

        return new RedirectResponse($trackBackURL);
    }
}
