<?php declare(strict_types=1);

namespace MagmodulesWebshopnl;

use Doctrine\DBAL\Connection;
use Shopware\Core\Defaults;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Plugin;
use Shopware\Core\Framework\Plugin\Context\InstallContext;
use Shopware\Core\Framework\Plugin\Context\UninstallContext;
use Shopware\Core\Framework\Plugin\Util\PluginIdProvider;
use Shopware\Core\Framework\Uuid\Uuid;

class MagmodulesWebshopnl extends Plugin
{
    const SHIPPING_METHOD_NAME = 'webshopnl';

    public function install(InstallContext $installContext): void
    {
        parent::install($installContext);
        $this->createFolder();
        $this->addConfigurationToken();
        $this->addPaymentMethod($installContext->getContext());
        $this->addShippingMethod($installContext->getContext());
    }

    /**
     * @param UninstallContext $uninstallContext
     * @return void
     */
    public function uninstall(UninstallContext $uninstallContext): void
    {
        /* Keep UserData? Then do nothing here */
        if ($uninstallContext->keepUserData()) {
            return;
        }

        /**
         * @var Connection $connection
         */
        $connection = $this->container->get(Connection::class);

        $connection->executeStatement(
            'DELETE FROM system_config WHERE configuration_key LIKE :domain',
            [
                'domain' => '%MagmodulesWebshopnl.settings%',
            ]
        );
        $connection->executeStatement('DROP TABLE IF EXISTS `magmodules_wevshopnl_product_feed`');
        $connection->executeStatement('DROP TABLE IF EXISTS `magmodules_webshopnl_order`');

        $folderPath = getcwd().'/';
        $dir = "webshopnl";
        $this->deleteDirectory($dir);
    }

    public function deleteDirectory($dir) {

        if (!file_exists($dir)) {
            return true;
        }

        if (!is_dir($dir)) {
            return unlink($dir);
        }

        foreach (scandir($dir) as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }

            if (!$this->deleteDirectory($dir . DIRECTORY_SEPARATOR . $item)) {
                return false;
            }

        }

        return rmdir($dir);
    }

    public function createFolder()
    {
        $folderPath = getcwd().'/';
        mkdir($folderPath."webshopnl");
    }

    public function addShippingMethod(Context $context)
    {
        /**
         * @var EntityRepository $shippingMethodRepository
         */
        $shippingMethodRepository = $this->container->get('shipping_method.repository');

        $shippingMethodRepository->upsert([
            $this->createDataPayload($shippingMethodRepository, $context)
        ], $context);
    }

    private function addRule($rulesRepository, Context $context): string
    {
        $ruleIdInsert = Uuid::randomHex();
        $data = [
            'id' => $ruleIdInsert,
            'name' => 'All customers',
            'priority' => 1,
            'invalid' => false
        ];
        $rulesRepository->upsert([$data], $context);
        return $ruleIdInsert;
    }
    public function createDataPayload($shippingMethodRepository, $context): array
    {
        $rulesRepository = $this->container->get('rule.repository');

        $ruleId = $rulesRepository->searchIds(
            (new Criteria())->addFilter(new EqualsFilter('name', 'All customers')),
            $context
        )->firstId();

        if (!$ruleId) {
            $ruleId = $rulesRepository->searchIds(
                (new Criteria())->setLimit(1),
                $context
            )->firstId();
        }

        if ($ruleId === null) {
            $ruleId = $this->addRule($rulesRepository, $context);
        }

        $deliveryTimeRepository = $this->container->get('delivery_time.repository');

        $deliveryTimeEntityId = $deliveryTimeRepository->searchIds(
            (new Criteria())->setLimit(1),
            $context
        )->firstId();

        if ($deliveryTimeEntityId === null) {
            $deliveryTimeEntityId = $this->addDeliveryTime($deliveryTimeRepository, $context);
        }

        $id = $this->getExistingShippingMethodId($shippingMethodRepository, $context);

        return [
            'id' => $id ?? Uuid::randomHex(),
            'translations' => [
                'en-GB' => [
                    'name' => self::SHIPPING_METHOD_NAME
                ],
                'nl-NL' => [
                    'name' => self::SHIPPING_METHOD_NAME
                ],
                'de-DE' => [
                    'name' => self::SHIPPING_METHOD_NAME
                ]
            ],
            'active' => false,
            'availabilityRuleId' => $ruleId,
            'deliveryTimeId' => $deliveryTimeEntityId,
            'prices' => [
                [
                    'price' => 0,
                    'currencyId' => Defaults::CURRENCY,
                    'calculation' => 0,
                    'quantityStart' => 0,
                    'currencyPrice' => [
                        [
                            'currencyId' => Defaults::CURRENCY,
                            'net' => 0,
                            'gross' => 0,
                            'linked' => false,
                        ],
                    ],
                ],
            ]
        ];
    }

    private function getExistingShippingMethodId($shippingMethodRepository, $context): ?string
    {
        $criteria = (new Criteria())
            ->addFilter(new EqualsFilter('name', self::SHIPPING_METHOD_NAME));

        return $shippingMethodRepository->searchIds($criteria, $context)->firstId();
    }

    private function addDeliveryTime($deliveryTimeRepository, Context $context): string
    {
        $deliveryIdInsert = Uuid::randomHex();

        $deliveryTimeRepository->upsert([
            [
                'id' => $deliveryIdInsert,
                'name' => 'All customers',
                'min' => 1,
                'max' => 5,
                'unit' => 'day'
            ]
        ], $context);

        return $deliveryIdInsert;
    }
    public function addPaymentMethod(Context $context): void
    {
        $paymentMethodExists = $this->getPaymentMethodId($context);

        /* Payment method exists already, no need to continue here */
        if ($paymentMethodExists) {
            return;
        }

        /**
         * @var PluginIdProvider $pluginIdProvider
         */
        $pluginIdProvider = $this->container->get(PluginIdProvider::class);
        $pluginId = $pluginIdProvider->getPluginIdByBaseClass(static::class, $context);

        $webshopnlData = [
            'name' => 'webshopnl',
            'pluginId' => $pluginId,
            'afterOrderEnabled' => false,
            'translations' => [
                'de-DE' => [
                    'name' => 'webshopnl',
                    'description' => 'webshopnl',
                ],
                'en-GB' => [
                    'name' => 'webshopnl',
                    'description' => 'webshopnl',
                ]
            ]
        ];

        /**
         * @var EntityRepository $paymentRepository
         */
        $paymentRepository = $this->container->get('payment_method.repository');
        $paymentRepository->create([$webshopnlData], $context);
    }

    private function getPaymentMethodId(Context $context): ?string
    {
        /**
         * @var EntityRepository $paymentRepository
         */
        $paymentRepository = $this->container->get('payment_method.repository');

        /* Fetch ID for update */
        $paymentCriteria = (new Criteria())->addFilter(new EqualsFilter('name','webshopnl'));
        $paymentIds = $paymentRepository->searchIds($paymentCriteria, $context);

        if ($paymentIds->getTotal() === 0) {
            return null;
        }

        return $paymentIds->getIds()[0];
    }
    private function addConfigurationToken()
    {
        $config = $this->container->get('Shopware\Core\System\SystemConfig\SystemConfigService');

        $token = $this->getRandomStringRand(25);
        $config->set('MagmodulesWebshopnl.settings.integrationToken', $token);
    }

    function getRandomStringRand($length)
    {
        $stringSpace = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $stringLength = strlen($stringSpace);
        $randomString = '';
        for ($i = 0; $i < $length; $i ++) {
            $randomString = $randomString . $stringSpace[rand(0, $stringLength - 1)];
        }
        return $randomString;
    }
}