<?php declare(strict_types=1);

namespace MagmodulesWebshopnl\Core\Api\Controller\Info;

use Composer\InstalledVersions;
use Shopware\Core\Framework\Context;
use Shopware\Core\Framework\DataAbstractionLayer\EntityRepository;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Criteria;
use Shopware\Core\Framework\DataAbstractionLayer\Search\Filter\EqualsFilter;
use Shopware\Core\Framework\Store\Services\InstanceService;
use Shopware\Core\System\SystemConfig\SystemConfigService;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @Route(defaults={"_routeScope"={"storefront"}})
 */
class InfoController extends AbstractController
{
    /**
     * @var EntityRepository
     */
    public $webshopnlProductFeedRepository;

    /**
     * @var SystemConfigService
     */
    public $systemConfigService;

    /**
     * @var InstanceService
     */
    public $instanceService;

    /**
     * @var EntityRepository
     */
    private $pluginRepository;

    public function __construct(
        EntityRepository $webshopnlProductFeedRepository,
        SystemConfigService $systemConfigService,
        EntityRepository $pluginRepository
    ) {
        $this->webshopnlProductFeedRepository= $webshopnlProductFeedRepository;
        $this->systemConfigService= $systemConfigService;
        $this->pluginRepository = $pluginRepository;
    }

    /**
     * @Route("/api/webshopnl/info",
     *     name="api.webshopnl.info",  methods={"GET"},
     *     defaults={"auth_required"=false, "_routeScope"={"administration"}})
     */
    public function getInfo(Request $request, Context $context): JsonResponse
    {
        $bearerToken = $request->headers->get('Authorization');
        if ($bearerToken) {
            $bearerToken = str_replace("Bearer ","",$bearerToken);
            $integrationToken = $this->systemConfigService->get('MagmodulesWebshopnl.settings.integrationToken');
            if ($bearerToken === $integrationToken) {

                $data = $this->getProductFeedData($context);
                
                return new JsonResponse([
                    'enabled' => $this->systemConfigService->get('MagmodulesWebshopnl.settings.enable'),
                    'shopware_version' => $this->getShopwareVersion(),
                    "plugin_version"=> $this->pluginData($context)->getVersion(),
                    "installation_id" => $this->pluginData($context)->getId(),
                    'feed_url' => $this->getFeedUrl($data),
                    'order_post_url' => $request->server->get('APP_URL').'/api/webshopnl/order',
                    'last_order_import' => 'dassadas',
                    'active_sales_channel' => $this->getActiveStore($data)
                ]);
            } else {
                return new JsonResponse([
                    'status' => 401,
                    'title' => 'The resource owner or authorization server denied the request',
                    "detail"=> 'Access token could not be verified',
                ]);
            }
        } else {
            return new JsonResponse([
                "status" => 401,
                "title" => 'The resource owner or authorization server denied the request.',
                "detail" => 'Missing Authorization header',
            ]);
        }
    }

    public function getShopwareVersion()
    {
        if (InstalledVersions::isInstalled('shopware/platform')) {
            return InstalledVersions::getVersion('shopware/platform');
        } else {
            return InstalledVersions::getVersion('shopware/core');
        }
    }

    public function pluginData($context)
    {
        $nameSpaceExplode = explode('\\', __NAMESPACE__);
        $pluginName = $nameSpaceExplode[0];

        $criteria = new Criteria();
        $criteria->addFilter(new EqualsFilter('plugin.name', $pluginName));
        return $this->pluginRepository->search($criteria, $context)->first();
    }
    public function getProductFeedData($context)
    {
        $criteria = new Criteria();
        $criteria->addAssociation('salesChannel');
        $criteria->addFilter(new EqualsFilter('salesChannel.active', true));
        return $this->webshopnlProductFeedRepository->search($criteria, $context);
    }

    public function getFeedUrl($data) : array
    {
        $feedUrls = [];
        foreach ($data as $feedData) {
            $feedUrls[] = [
                'salesChannel' => $feedData->getId(),
                'url' => $feedData->getProductFeedUrl()
            ];
        }
        return $feedUrls;
    }

    public function getActiveStore($data) : array
    {
        $storeIds = [];
        foreach ($data as $feedData) {
            if ($this->systemConfigService->get('MagmodulesWebshopnl.settings.enable', $feedData->getSalesChannelId())) {
                $storeIds[] = [
                    'integration' => $this->systemConfigService->get('MagmodulesWebshopnl.settings.enable', $feedData->getSalesChannelId()),
                    'sales_channel_id' => $feedData->getSalesChannelId(),
                    'store_name' => $feedData->getSalesChannel()->getName()
                ];
            }
        }

        return $storeIds;
    }
}

